function [incli,Nmin,NPmin,beamwidthT_edge,cellNumber,Dedge,period,cellDuration,DSatT,Afoot,wavelength] =...
    constellation(constel,alti,elev,diversity,GTedge_dB,freq)
% Program that calculates coverage geometry for different LEO constellations
% Darren Chang (darrenz@mit.edu)
% Last modified by 7/25/03
% This program calculates orbit plane number and
% number of sats. per plane that will provide the
% minimum total amount of sats. for a global coverage
% at given satellite constellation type, altitude,
% and elevation angle.  The calculation can
% be done either for Walker constellation or Polar
% constellation.
% The equations for Polar constellation are based on Chapter 2 and 3 in
% LWJ00.
% Unless followed by _d, otherwise all angles are in radian.
% orbitR   - earth-centered sat. radius (km)
% Dsat     - satellite antenna diameter (m)
% freq     - frequency (Hz)
% aperture - aperture type (1 for tapered and 2 for uniform)
% constel  - constellation type (1 for polar and 2 for Walker)
% elev     - elevation angle
% N		  - number of satellites in constellation
% NS		  - number of satellites per plane
% NP		  - number of planes
Rearth   = 6371;  % Earth average radius in km
Nmin = 1000000;   % set Nmin to a large number

orbitR     = alti + Rearth;
nadir = asin(Rearth/orbitR*cos(elev));					% nadir angle in radian, eqn (2.24)
central = acos(Rearth/orbitR*cos(elev)) - elev;	        % earth central angle in radian, eqn (2.23)
central0= acos(Rearth/orbitR*cos(0)) - 0;	            % earth central angle in radian when elevation = 0

% polar constellation
if constel == 1
    incli = pi/2;							    % inclination angle
for NP = 1:1000									% try up to 1000 planes
  tmp = (pi - (NP - 1)*central)/(NP + 1);
  if cos(central) > cos(tmp)					% if there are not enough planes to provide global coverage, then do nothing
      ;
  else
     NS = ceil(pi/acos(cos(central)/cos((pi - (NP - 1)*central)/(NP + 1))));
     N = NS*NP;
     if N < Nmin
        Nmin  = N;	    % minimum number of satellites in constellation
        NSmin = NS;	    % minimum number of satellites per plane
        NPmin = NP;	    % minimum number of planes
     else
        ;
     end; %end if
  end; %end if
end; %end for

% Nmin = Nmin*diversity;  % Diversity will increase number of sats. in a Polar constellation
% NPmin = NPmin*diversity;   % and increase the number of planes.

NPmin = ceil(NPmin*diversity);
Nmin = NPmin*NSmin;

% Walker constellation
elseif constel == 2
    [Nmin,NPmin,Fmin,incli] = walker_lang(central0*180/pi,diversity);   
else
   ;
end;

% Find number of cells per footprint
GTedge = 10^(GTedge_dB/10);
wavelength = 3e8/(freq*1e6);        % [m]
DSatT = wavelength*sqrt(GTedge)/pi;
beamwidthT_edge = 35*pi/sqrt(GTedge);        % [deg]. Based on equation (3.6) in Lutz.  

Afoot  = 2*pi*Rearth^2*(1 - cos(central));					% footprint area in km^2, eqn (2.31)
slantRange = sqrt(Rearth^2 + orbitR^2 - 2*Rearth*orbitR*cos(central));	% slant range in km, eqn (2.26)

R_footprint = Rearth*central;

alpha1 = nadir - 2*beamwidthT_edge*pi/180;            % [rad], alpha1 = the nadir angle - 2 beamwidth
beta1  = asin(orbitR/Rearth*sin(alpha1)) - alpha1;    % [rad], the corresponding earth centered angle to alpha1
gamma1 = central - beta1;                             % [rad], the earth centered angle corresponding to edge cell diameter
edgecell_r = 0.5*Rearth*gamma1;
edgecell_A = 2*pi*Rearth^2*(1 - cos(gamma1/2));
edgehexa_A = 3/pi*sin(pi/3)*edgecell_A;

GTcenter = GTedge_dB - 6;
beamwidthT_center = 35*pi/sqrt(GTcenter);
alpha2 = beamwidthT_center*pi/180;
if orbitR/Rearth*sin(alpha2) > 1
    beta2 = central;
    cellNumber = 1;
else
    beta2  = asin(orbitR/Rearth*sin(alpha2)) - alpha2;
	centercell_r = Rearth*beta2;
	centercell_A = 2*pi*Rearth^2*(1 - cos(beta2));
	centerhexa_A = 3/pi*sin(pi/3)*centercell_A;
	
	n = 1.21*(1 - cos(nadir))/(1 - cos(beamwidthT_edge*pi/180)); % cell # estimation used by Lutz. Not used here.
	
	cellNumber = round(Afoot/((centerhexa_A + edgehexa_A)/2))*2;
end;

alpha3 = nadir - beamwidthT_edge*pi/180;              % [rad], alpha3 = the nadir angle - beamwidth
beta3  = asin(orbitR/Rearth*sin(alpha3)) - alpha3;    % [rad], the corresponding earth centered angle to alpha3
% distance from sat to the center of the edge cell
Dedge = sqrt(Rearth^2 + orbitR^2 - 2*Rearth*orbitR*cos(beta3));

% Dmiddle is distance from sat to the central angle bisecting point on the ground
Dmiddle = sqrt(Rearth^2 + orbitR^2 - 2*Rearth*orbitR*cos(central/2));

% find orbital period
GMearth=398601;
period=2*pi*sqrt((orbitR)^3/GMearth)/60;    %[minutes], orbital period

% find duration of a beam over a center cell
Wcir = 631.34812*(orbitR)^-1.5;             % [rad/s], orbital angular velocity
centercell_Beta = 2*beta2;                   % [rad], central angle of a center cell
cellDuration = centercell_Beta/Wcir;        % [sec], duration of a beam over a center cell
visibleDuration = 2*central/Wcir/60;        % [min], duration of a satellite visible to the user above min. elevation